import { Helmet } from 'react-helmet-async';
import { motion } from 'framer-motion';
import { Check, ChevronRight, Zap, Shield, Globe } from 'lucide-react';
import Navbar from './Navbar';
import Footer from './Footer';

const plans = [
    {
        name: "Base Plan",
        description: "Essential compliance for small facilities.",
        price: "Contact Sales",
        features: [
            "Carbon Center",
            "ISO Core",
            "Environmental Compliance",
            "Basic Reporting",
            "Up to 5 Users"
        ],
        icon: Zap,
        color: "text-accent",
        bg: "bg-accent/10",
        border: "border-accent/20",
        popular: false
    },
    {
        name: "Pro Plan",
        description: "Advanced analytics for growing enterprises.",
        price: "Custom Quote",
        features: [
            "Everything in Base",
            "ESG Management",
            "LEED Tracking",
            "Safety & Fire Modules",
            "IoT Basic Integration",
            "Advanced Analytics"
        ],
        icon: Shield,
        color: "text-primary",
        bg: "bg-primary/10",
        border: "border-primary/50",
        popular: true
    },
    {
        name: "Enterprise Plan",
        description: "Full-scale digital transformation & digital twin.",
        price: "Tailored",
        features: [
            "Everything in Pro",
            "Digital Twin Technology",
            "IoT Advanced Integration",
            "Full Customization",
            "24/7 Dedicated Support",
            "API Access"
        ],
        icon: Globe,
        color: "text-purple-400",
        bg: "bg-purple-400/10",
        border: "border-purple-400/20",
        popular: false
    }
];

export default function PricingPage() {
    return (
        <div className="bg-dark min-h-screen text-white overflow-x-hidden">
            <Helmet>
                <title>Pricing - Mutqin Pro</title>
                <meta name="description" content="Flexible pricing plans for Mutqin Pro. Choose the right compliance and sustainability package for your organization." />
            </Helmet>
            <Navbar />

            {/* Hero Section */}
            <section className="relative pt-32 pb-20 overflow-hidden">
                <div className="absolute inset-0 bg-dark z-0">
                    <div className="absolute top-0 right-0 w-[600px] h-[600px] bg-primary/10 rounded-full blur-[128px]" />
                    <div className="absolute bottom-0 left-0 w-[600px] h-[600px] bg-secondary/20 rounded-full blur-[128px]" />
                </div>

                <div className="container mx-auto px-6 relative z-10 text-center">
                    <motion.div
                        initial={{ opacity: 0, y: 30 }}
                        animate={{ opacity: 1, y: 0 }}
                        transition={{ duration: 0.8 }}
                    >
                        <h1 className="text-4xl lg:text-6xl font-bold leading-tight mb-6">
                            Flexible Plans for <span className="text-gradient">Every Sector</span>
                        </h1>
                        <p className="text-xl text-gray-400 mb-10 leading-relaxed max-w-3xl mx-auto">
                            Choose a plan that fits your compliance, sustainability, and operational needs — scalable for factories, enterprises, and government entities.
                        </p>
                        <motion.button
                            whileHover={{ scale: 1.05 }}
                            whileTap={{ scale: 0.95 }}
                            className="bg-primary hover:bg-primary/90 text-white px-8 py-4 rounded-full font-bold text-lg transition-all shadow-lg shadow-primary/30 flex items-center gap-2 mx-auto"
                        >
                            Request Demo
                            <ChevronRight className="w-5 h-5" />
                        </motion.button>
                    </motion.div>
                </div>
            </section>

            {/* Pricing Cards */}
            <section className="py-20 relative">
                <div className="container mx-auto px-6">
                    <div className="grid md:grid-cols-3 gap-8 max-w-6xl mx-auto">
                        {plans.map((plan, index) => (
                            <motion.div
                                key={index}
                                initial={{ opacity: 0, y: 30 }}
                                whileInView={{ opacity: 1, y: 0 }}
                                viewport={{ once: true }}
                                transition={{ delay: index * 0.1 }}
                                whileHover={{ y: -10 }}
                                className={`relative bg-white/5 backdrop-blur-lg border rounded-3xl p-8 transition-all duration-300 flex flex-col ${plan.popular ? 'border-primary shadow-glow scale-105 z-10' : 'border-white/10 hover:border-white/20'}`}
                            >
                                {plan.popular && (
                                    <div className="absolute -top-4 left-1/2 -translate-x-1/2 bg-primary text-white px-4 py-1 rounded-full text-sm font-bold shadow-lg">
                                        Most Popular
                                    </div>
                                )}

                                <div className={`w-14 h-14 rounded-2xl ${plan.bg} ${plan.border} border flex items-center justify-center mb-6`}>
                                    <plan.icon className={`w-7 h-7 ${plan.color}`} />
                                </div>

                                <h3 className="text-2xl font-bold mb-2">{plan.name}</h3>
                                <p className="text-gray-400 text-sm mb-6 h-10">{plan.description}</p>

                                <div className="text-3xl font-bold mb-8 text-white">{plan.price}</div>

                                <div className="space-y-4 mb-8 flex-1">
                                    {plan.features.map((feature, i) => (
                                        <div key={i} className="flex items-center gap-3 text-sm text-gray-300">
                                            <div className={`w-5 h-5 rounded-full ${plan.bg} flex items-center justify-center flex-shrink-0`}>
                                                <Check className={`w-3 h-3 ${plan.color}`} />
                                            </div>
                                            {feature}
                                        </div>
                                    ))}
                                </div>

                                <button className={`w-full py-3 rounded-xl font-medium transition-all ${plan.popular ? 'bg-primary hover:bg-primary/90 text-white shadow-lg shadow-primary/25' : 'bg-white/5 hover:bg-white/10 text-white border border-white/10'}`}>
                                    Get Started
                                </button>
                            </motion.div>
                        ))}
                    </div>
                </div>
            </section>

            {/* Support Section */}
            <section className="py-24 bg-dark/50 relative border-t border-white/5">
                <div className="container mx-auto px-6 text-center">
                    <motion.div
                        initial={{ opacity: 0, y: 20 }}
                        whileInView={{ opacity: 1, y: 0 }}
                        viewport={{ once: true }}
                    >
                        <h2 className="text-3xl font-bold mb-4">Need a Custom Solution?</h2>
                        <p className="text-gray-400 mb-8 max-w-2xl mx-auto">
                            Not sure which plan fits your organization? Contact us for a tailored consultation and see how Mutqin Pro can transform your operations.
                        </p>
                        <div className="flex flex-col sm:flex-row items-center justify-center gap-4">
                            <button className="bg-primary hover:bg-primary/90 text-white px-8 py-3 rounded-2xl font-medium transition-all shadow-lg shadow-primary/20">
                                Request Demo
                            </button>
                            <button className="bg-white/5 hover:bg-white/10 text-white border border-white/10 px-8 py-3 rounded-2xl font-medium transition-all">
                                Contact Sales
                            </button>
                        </div>
                    </motion.div>
                </div>
            </section>

            <Footer />
        </div>
    );
}
