import { useState } from 'react';
import { motion } from 'framer-motion';
import {
    Leaf, FileCheck, ShieldCheck, BarChart3,
    Recycle, Building2, HardHat, Scale,
    ArrowRight, CheckCircle2, AlertTriangle
} from 'lucide-react';
import Navbar from './Navbar';
import Footer from './Footer';

const modules = [
    {
        icon: Leaf,
        title: "Carbon Center",
        description: "Automates carbon and emissions management across Scope 1, 2, and 3.",
        howItWorks: "Collects data from IoT meters, ERP, utilities; analyzes emissions; generates GHG & NCEC reports.",
        value: "Accurate carbon insights, reduction planning, regulatory compliance.",
        color: "text-green-400",
        bg: "bg-green-400/10",
        border: "border-green-400/20"
    },
    {
        icon: FileCheck,
        title: "ISO Compliance",
        description: "All ISO documents, audits, NCRs, and CAPA actions in one place.",
        howItWorks: "Centralized document control → audit workflows → corrective actions → ISO-compliant reports.",
        value: "Zero missing documents, smoother audits, real-time compliance status.",
        color: "text-blue-400",
        bg: "bg-blue-400/10",
        border: "border-blue-400/20"
    },
    {
        icon: ShieldCheck,
        title: "Environmental Compliance",
        description: "Manage inspections, monitoring, permits, and environmental obligations.",
        howItWorks: "Logs inspections, tracks water/air/waste readings, schedules recurring checks.",
        value: "Continuous compliance with NCEC and environmental regulations.",
        color: "text-primary",
        bg: "bg-primary/10",
        border: "border-primary/20"
    },
    {
        icon: BarChart3,
        title: "EcoVadis & ESG",
        description: "Complete ESG scoring, GRI framework alignment, and gap analysis.",
        howItWorks: "Imports ESG data, maps SDG goals, performs auto gap analysis vs EcoVadis criteria.",
        value: "Higher ESG ratings, actionable improvement roadmap.",
        color: "text-purple-400",
        bg: "bg-purple-400/10",
        border: "border-purple-400/20"
    },
    {
        icon: Recycle,
        title: "LCA / EPD",
        description: "Automate Life Cycle Assessments and generate EPD reports.",
        howItWorks: "Product data → LCA engine (GWP, energy, water) → formatted EPD PDF export.",
        value: "Faster certifications and full environmental transparency.",
        color: "text-yellow-400",
        bg: "bg-yellow-400/10",
        border: "border-yellow-400/20"
    },
    {
        icon: Building2,
        title: "Green Building (LEED)",
        description: "Track LEED points, energy consumption, water usage, air quality.",
        howItWorks: "Pulls data from BMS/IoT → updates LEED score → generates compliance reports.",
        value: "Improve building sustainability performance and meet certification criteria.",
        color: "text-emerald-400",
        bg: "bg-emerald-400/10",
        border: "border-emerald-400/20"
    },
    {
        icon: HardHat,
        title: "Safety & Fire",
        description: "Central hub for safety inspections, fire equipment checks, incident logs.",
        howItWorks: "Schedules inspections, captures results, assigns recommendations.",
        value: "Better safety readiness, fewer violations, complete audit trail.",
        color: "text-orange-400",
        bg: "bg-orange-400/10",
        border: "border-orange-400/20"
    },
    {
        icon: Scale,
        title: "Government Compliance",
        description: "One place for HCIS, SASO, NCEC, Civil Defense, Balady requirements.",
        howItWorks: "Tracks obligations, deadlines, evidence, and audit submissions.",
        value: "Zero missed deadlines, instant compliance visibility.",
        color: "text-red-400",
        bg: "bg-red-400/10",
        border: "border-red-400/20"
    }
];

import { Helmet } from 'react-helmet-async';

export default function SolutionsPage() {
    const [activePreview, setActivePreview] = useState(0);

    return (
        <div className="bg-dark min-h-screen text-white overflow-x-hidden">
            <Helmet>
                <title>Solutions - Mutqin Pro Modules</title>
                <meta name="description" content="Explore Mutqin Pro's specialized modules for Carbon Management, ISO IMS, Environmental Compliance, ESG, and more. Unified in one platform." />
            </Helmet>
            <Navbar />

            {/* Hero Section */}
            <section className="relative pt-32 pb-20 overflow-hidden">
                <div className="absolute inset-0 bg-dark z-0">
                    <div className="absolute top-0 right-0 w-[600px] h-[600px] bg-primary/10 rounded-full blur-[128px]" />
                    <div className="absolute bottom-0 left-0 w-[600px] h-[600px] bg-secondary/20 rounded-full blur-[128px]" />
                </div>

                <div className="container mx-auto px-6 relative z-10 grid lg:grid-cols-2 gap-12 items-center">
                    <motion.div
                        initial={{ opacity: 0, x: -30 }}
                        animate={{ opacity: 1, x: 0 }}
                        transition={{ duration: 0.8 }}
                    >
                        <h1 className="text-4xl lg:text-6xl font-bold leading-tight mb-6">
                            All Your Compliance & Sustainability Modules <br />
                            <span className="text-gradient">Unified in One Platform</span>
                        </h1>
                        <p className="text-xl text-gray-400 mb-8 leading-relaxed">
                            Explore each module and see how Mutqin Pro automates data collection, analytics, reporting, and regulatory alignment—powered by AI.
                        </p>
                    </motion.div>

                    <motion.div
                        initial={{ opacity: 0, scale: 0.8 }}
                        animate={{ opacity: 1, scale: 1 }}
                        transition={{ duration: 1 }}
                        className="relative h-[500px] flex items-center justify-center"
                    >
                        {/* Central Logo with Glow */}
<div className="absolute w-72 h-72 z-40 flex items-center justify-center">


    {/* صورة اللوقو */}
    <img
        src="/logo-icon.png"   // غير المسار حسب مكان اللوقو
        alt="MUTQIN Logo"
        className="relative z-20 w-32 h-32 object-contain rounded-xl shadow-glow"
    />
</div>

                        {/* Orbiting Cards */}
                        {[0, 1, 2, 3, 4, 5, 6, 7].map((i) => (
                            <motion.div
                                key={i}
                                animate={{ rotate: 360 }}
                                transition={{ duration: 40, repeat: Infinity, ease: "linear" }}
                                className="absolute w-full h-full"
                            >
                                <div
                                    className="absolute top-0 left-1/2 -translate-x-1/2 -translate-y-1/2 w-16 h-16 bg-dark/80 backdrop-blur border border-white/10 rounded-xl flex items-center justify-center shadow-lg transform"
                                    style={{ transform: `rotate(${i * 45}deg) translateY(-180px) rotate(-${i * 45}deg)` }}
                                >
                                    <div className={`w-3 h-3 rounded-full ${i % 2 === 0 ? 'bg-primary' : 'bg-accent'}`} />
                                </div>
                            </motion.div>
                        ))}

                        {/* Orbit Rings */}
                        <div className="absolute inset-0 border border-white/5 rounded-full" />
                        <div className="absolute inset-12 border border-white/5 rounded-full border-dashed" />
                    </motion.div>
                </div>
            </section>

            {/* Modules Grid */}
            <section className="py-20 relative">
                <div className="container mx-auto px-6">
                    <div className="grid md:grid-cols-2 lg:grid-cols-3 gap-8">
                        {modules.map((module, index) => (
                            <motion.div
                                key={index}
                                initial={{ opacity: 0, y: 30 }}
                                whileInView={{ opacity: 1, y: 0 }}
                                viewport={{ once: true }}
                                transition={{ delay: index * 0.1 }}
                                whileHover={{ y: -10 }}
                                className="group relative bg-white/5 backdrop-blur-lg border border-white/10 rounded-3xl p-8 hover:border-primary/50 transition-all duration-300"
                            >
                                <div className={`w-14 h-14 rounded-2xl ${module.bg} ${module.border} border flex items-center justify-center mb-6 group-hover:scale-110 transition-transform`}>
                                    <module.icon className={`w-7 h-7 ${module.color}`} />
                                </div>

                                <h3 className="text-2xl font-bold mb-3 group-hover:text-primary transition-colors">{module.title}</h3>
                                <p className="text-gray-400 mb-4 text-sm">{module.description}</p>

                                <div className="space-y-3 mb-6">
                                    <div className="text-xs text-gray-500 uppercase tracking-wider font-semibold">How it works</div>
                                    <p className="text-sm text-gray-300 leading-relaxed">{module.howItWorks}</p>

                                    <div className="text-xs text-gray-500 uppercase tracking-wider font-semibold mt-4">Value Delivered</div>
                                    <p className="text-sm text-gray-300 leading-relaxed">{module.value}</p>
                                </div>

                                <button
                                    onClick={() => {
                                        const element = document.getElementById('dashboard-preview');
                                        element?.scrollIntoView({ behavior: 'smooth' });
                                        setActivePreview(index % 3); // Map to 3 previews
                                    }}
                                    className="w-full py-3 rounded-xl bg-white/5 hover:bg-primary hover:text-white border border-white/10 transition-all text-sm font-medium flex items-center justify-center gap-2"
                                >
                                    Learn More
                                    <ArrowRight className="w-4 h-4" />
                                </button>
                            </motion.div>
                        ))}
                    </div>
                </div>
            </section>

            {/* Dashboard Preview Section */}
            <section id="dashboard-preview" className="py-24 bg-dark/50 relative">
                <div className="container mx-auto px-6">
                    <motion.div
                        initial={{ opacity: 0, y: 20 }}
                        whileInView={{ opacity: 1, y: 0 }}
                        viewport={{ once: true }}
                        className="text-center mb-16"
                    >
                        <h2 className="text-4xl font-bold mb-4">Interactive <span className="text-gradient">Dashboard Snapshots</span></h2>
                        <p className="text-gray-400">Real-time visibility into your compliance and sustainability performance.</p>
                    </motion.div>

                    <div className="grid lg:grid-cols-3 gap-8 mb-12">
                        {['Carbon Center', 'ISO IMS', 'Environmental Compliance'].map((name, i) => (
                            <button
                                key={i}
                                onClick={() => setActivePreview(i)}
                                className={`p-4 rounded-xl border transition-all text-left ${activePreview === i
                                    ? 'bg-primary/10 border-primary text-white'
                                    : 'bg-white/5 border-white/10 text-gray-400 hover:bg-white/10'
                                    }`}
                            >
                                <div className="font-bold mb-1">{name}</div>
                                <div className="text-xs opacity-70">Click to view dashboard</div>
                            </button>
                        ))}
                    </div>

                    <div className="relative aspect-video bg-dark border border-white/10 rounded-3xl overflow-hidden shadow-2xl">
                        {/* Dashboard Header */}
                        <div className="absolute top-0 left-0 right-0 h-12 bg-white/5 border-b border-white/10 flex items-center px-6 justify-between">
                            <div className="flex items-center gap-4">
                                <div className="flex gap-2">
                                    <div className="w-3 h-3 rounded-full bg-red-500" />
                                    <div className="w-3 h-3 rounded-full bg-yellow-500" />
                                    <div className="w-3 h-3 rounded-full bg-green-500" />
                                </div>
                                <div className="text-sm text-gray-400">Mutqin Pro / {['Carbon Center', 'ISO IMS', 'Environmental Compliance'][activePreview]}</div>
                            </div>
                        </div>

                        {/* Dashboard Content */}
                        <div className="absolute inset-0 top-12 p-6 bg-dark/80 backdrop-blur">
                            {activePreview === 0 && (
                                // Carbon Center Wireframe
                                <div className="grid grid-cols-4 gap-6 h-full">
                                    <div className="col-span-3 grid grid-rows-2 gap-6">
                                        <div className="bg-white/5 rounded-xl p-6 border border-white/5 relative overflow-hidden">
                                            <h4 className="text-sm text-gray-400 mb-4">Emissions Timeline (Scope 1, 2, 3)</h4>
                                            <div className="absolute bottom-0 left-0 right-0 h-32 flex items-end px-6 gap-2">
                                                {[40, 60, 45, 70, 50, 65, 55, 80, 60, 75, 50, 60].map((h, i) => (
                                                    <motion.div
                                                        key={i}
                                                        initial={{ height: 0 }}
                                                        animate={{ height: `${h}%` }}
                                                        transition={{ delay: i * 0.05 }}
                                                        className="flex-1 bg-gradient-to-t from-green-500/20 to-green-500 rounded-t-sm"
                                                    />
                                                ))}
                                            </div>
                                        </div>
                                        <div className="grid grid-cols-2 gap-6">
                                            <div className="bg-white/5 rounded-xl p-6 border border-white/5">
                                                <h4 className="text-sm text-gray-400 mb-2">Carbon Intensity</h4>
                                                <div className="text-3xl font-bold text-white">0.42 <span className="text-sm font-normal text-gray-500">kgCO2e/SAR</span></div>
                                                <div className="mt-4 h-2 bg-white/10 rounded-full overflow-hidden">
                                                    <div className="h-full w-[70%] bg-yellow-500" />
                                                </div>
                                            </div>
                                            <div className="bg-white/5 rounded-xl p-6 border border-white/5">
                                                <h4 className="text-sm text-gray-400 mb-2">Reduction Target</h4>
                                                <div className="text-3xl font-bold text-white">-15% <span className="text-sm font-normal text-gray-500">by 2025</span></div>
                                                <div className="mt-4 text-xs text-green-400">On track</div>
                                            </div>
                                        </div>
                                    </div>
                                    <div className="col-span-1 bg-white/5 rounded-xl p-6 border border-white/5">
                                        <h4 className="text-sm text-gray-400 mb-4">Emission Sources</h4>
                                        <div className="space-y-4">
                                            {['Electricity', 'Fleet', 'Manufacturing', 'Waste'].map((item, i) => (
                                                <div key={i}>
                                                    <div className="flex justify-between text-xs mb-1">
                                                        <span>{item}</span>
                                                        <span>{80 - i * 15}%</span>
                                                    </div>
                                                    <div className="h-1.5 bg-white/10 rounded-full overflow-hidden">
                                                        <div className="h-full bg-primary" style={{ width: `${80 - i * 15}%` }} />
                                                    </div>
                                                </div>
                                            ))}
                                        </div>
                                    </div>
                                </div>
                            )}

                            {activePreview === 1 && (
                                // ISO IMS Wireframe
                                <div className="grid grid-cols-3 gap-6 h-full">
                                    <div className="col-span-1 space-y-6">
                                        <div className="bg-white/5 rounded-xl p-6 border border-white/5">
                                            <h4 className="text-sm text-gray-400 mb-4">Document Status</h4>
                                            <div className="flex items-center justify-center py-4">
                                                <div className="w-32 h-32 rounded-full border-8 border-white/10 border-t-blue-500 border-r-blue-500 rotate-45" />
                                            </div>
                                            <div className="flex justify-center gap-4 text-xs">
                                                <span className="flex items-center gap-1"><div className="w-2 h-2 bg-blue-500 rounded-full" /> Active</span>
                                                <span className="flex items-center gap-1"><div className="w-2 h-2 bg-white/10 rounded-full" /> Draft</span>
                                            </div>
                                        </div>
                                        <div className="bg-white/5 rounded-xl p-6 border border-white/5">
                                            <div className="flex justify-between items-center mb-2">
                                                <h4 className="text-sm text-gray-400">Open NCRs</h4>
                                                <span className="text-red-400 font-bold">3</span>
                                            </div>
                                            <div className="space-y-2">
                                                {[1, 2, 3].map((i) => (
                                                    <div key={i} className="text-xs bg-red-500/10 text-red-300 p-2 rounded border border-red-500/20">
                                                        NCR-2024-00{i}: Safety equipment missing
                                                    </div>
                                                ))}
                                            </div>
                                        </div>
                                    </div>
                                    <div className="col-span-2 bg-white/5 rounded-xl p-6 border border-white/5">
                                        <h4 className="text-sm text-gray-400 mb-4">Audit Schedule & Progress</h4>
                                        <div className="space-y-4">
                                            {['ISO 9001: Quality', 'ISO 14001: Environment', 'ISO 45001: Safety'].map((iso, i) => (
                                                <div key={i} className="flex items-center gap-4 p-3 bg-white/5 rounded-lg border border-white/5">
                                                    <FileCheck className="w-5 h-5 text-blue-400" />
                                                    <div className="flex-1">
                                                        <div className="text-sm font-medium">{iso}</div>
                                                        <div className="text-xs text-gray-500">Next Audit: Oct {15 + i * 5}, 2024</div>
                                                    </div>
                                                    <div className="px-3 py-1 rounded-full bg-green-500/10 text-green-400 text-xs border border-green-500/20">
                                                        Compliant
                                                    </div>
                                                </div>
                                            ))}
                                        </div>
                                    </div>
                                </div>
                            )}

                            {activePreview === 2 && (
                                // Environmental Compliance Wireframe
                                <div className="grid grid-cols-3 gap-6 h-full">
                                    <div className="col-span-2 bg-white/5 rounded-xl p-6 border border-white/5">
                                        <h4 className="text-sm text-gray-400 mb-4">Monitoring Trends (Water & Air)</h4>
                                        <div className="h-48 flex items-end gap-1">
                                            {[...Array(20)].map((_, i) => (
                                                <div key={i} className="flex-1 flex flex-col justify-end gap-1">
                                                    <div className="w-full bg-blue-500/50 rounded-sm" style={{ height: `${Math.random() * 40 + 20}%` }} />
                                                    <div className="w-full bg-cyan-400/50 rounded-sm" style={{ height: `${Math.random() * 30 + 10}%` }} />
                                                </div>
                                            ))}
                                        </div>
                                        <div className="flex gap-4 mt-4 text-xs justify-center">
                                            <span className="flex items-center gap-1"><div className="w-2 h-2 bg-blue-500 rounded-full" /> Water Usage</span>
                                            <span className="flex items-center gap-1"><div className="w-2 h-2 bg-cyan-400 rounded-full" /> Air Quality Index</span>
                                        </div>
                                    </div>
                                    <div className="col-span-1 space-y-6">
                                        <div className="bg-white/5 rounded-xl p-6 border border-white/5">
                                            <h4 className="text-sm text-gray-400 mb-4">Compliance Status</h4>
                                            <div className="space-y-3">
                                                <div className="flex justify-between items-center p-2 bg-green-500/10 rounded border border-green-500/20">
                                                    <span className="text-sm">NCEC Permit</span>
                                                    <CheckCircle2 className="w-4 h-4 text-green-500" />
                                                </div>
                                                <div className="flex justify-between items-center p-2 bg-yellow-500/10 rounded border border-yellow-500/20">
                                                    <span className="text-sm">Waste Log</span>
                                                    <AlertTriangle className="w-4 h-4 text-yellow-500" />
                                                </div>
                                                <div className="flex justify-between items-center p-2 bg-green-500/10 rounded border border-green-500/20">
                                                    <span className="text-sm">Noise Level</span>
                                                    <CheckCircle2 className="w-4 h-4 text-green-500" />
                                                </div>
                                            </div>
                                        </div>
                                        <div className="bg-white/5 rounded-xl p-6 border border-white/5">
                                            <h4 className="text-sm text-gray-400 mb-2">Next Inspection</h4>
                                            <div className="text-xl font-bold">Dec 12, 2024</div>
                                            <div className="text-xs text-gray-500">Civil Defense Routine Check</div>
                                        </div>
                                    </div>
                                </div>
                            )}
                        </div>
                    </div>
                </div>
            </section>

            {/* CTA Section */}
            <section className="py-24 bg-gradient-to-b from-dark to-primary/10">
                <div className="container mx-auto px-6 text-center">
                    <h2 className="text-4xl font-bold mb-6">Discover how each module transforms your <br />compliance and sustainability workflow.</h2>
                    <motion.button
                        whileHover={{ scale: 1.05 }}
                        whileTap={{ scale: 0.95 }}
                        className="bg-primary hover:bg-primary/90 text-white px-10 py-5 rounded-full font-bold text-xl transition-all shadow-lg shadow-primary/30"
                    >
                        Request Full Demo
                    </motion.button>
                </div>
            </section>

            <Footer />
        </div>
    );
}
