import { Database, Radio, MapPin, Upload } from 'lucide-react';
import { ComponentLibraryItem, ModuleType } from '../../types';
import { useComponentDraggable } from '../dnd/useDragAndDrop';

const componentItems: ComponentLibraryItem[] = [
  {
    id: 'lib-simulation',
    type: ModuleType.Simulation,
    title: 'Simulation Data Generator',
    icon: 'database',
    description: 'Generate sample emission data',
  },
  {
    id: 'lib-sensor',
    type: ModuleType.Sensor,
    title: 'Sensor Type Selector',
    icon: 'radio',
    description: 'Select and configure sensor types',
  },
  {
    id: 'lib-floorzone',
    type: ModuleType.FloorZone,
    title: 'Floor & Zone Mapper',
    icon: 'mappin',
    description: 'Map floors and zones',
  },
  {
    id: 'lib-upload',
    type: ModuleType.Upload,
    title: 'Data Upload Module',
    icon: 'upload',
    description: 'Upload CSV or JSON files',
  },
];

const iconMap: Record<string, any> = {
  database: Database,
  radio: Radio,
  mappin: MapPin,
  upload: Upload,
};

export function ComponentsLibrary() {
  return (
    <div className="w-64 border-r border-gray-200 bg-gray-50 p-4 overflow-y-auto">
      <h3 className="text-sm font-semibold text-gray-700 mb-4">Components Library</h3>
      <div className="space-y-3">
        {componentItems.map((item) => {
          const ComponentCard = () => {
            const { ref, style, attributes, listeners, isDragging } = useComponentDraggable(
              item.id,
              item.type
            );
            const Icon = iconMap[item.icon] || Database;

            return (
              <div
                ref={ref}
                style={style}
                {...attributes}
                {...listeners}
                className={`p-4 bg-white border-2 border-gray-200 rounded-lg cursor-grab active:cursor-grabbing hover:shadow-md transition-all ${
                  isDragging ? 'opacity-50' : ''
                }`}
              >
                <div className="flex items-start gap-3">
                  <div className="p-2 bg-primary/10 rounded-lg">
                    <Icon className="w-5 h-5 text-primary" />
                  </div>
                  <div className="flex-1 min-w-0">
                    <h4 className="text-sm font-semibold text-gray-800">{item.title}</h4>
                    <p className="text-xs text-gray-500 mt-1">{item.description}</p>
                  </div>
                </div>
              </div>
            );
          };

          return <ComponentCard key={item.id} />;
        })}
      </div>
    </div>
  );
}

