import { X, ChevronDown, ChevronUp, Settings } from 'lucide-react';
import { Module } from '../../types';
import { useInputConfig } from '../../store';

interface ModuleCardProps {
  module: Module;
  onSelect: () => void;
}

export function ModuleCard({ module, onSelect }: ModuleCardProps) {
  const { removeModule, toggleModuleCollapse } = useInputConfig();

  const moduleTitles: Record<string, string> = {
    simulation: 'Simulation Data Generator',
    sensor: 'Sensor Type Selector',
    floorzone: 'Floor & Zone Mapper',
    upload: 'Data Upload Module',
  };

  const moduleColors: Record<string, string> = {
    simulation: 'bg-blue-50 border-blue-200 text-blue-700',
    sensor: 'bg-green-50 border-green-200 text-green-700',
    floorzone: 'bg-purple-50 border-purple-200 text-purple-700',
    upload: 'bg-orange-50 border-orange-200 text-orange-700',
  };

  const handleDelete = (e: React.MouseEvent) => {
    e.stopPropagation();
    removeModule(module.floorId, module.position);
  };

  const handleToggleCollapse = (e: React.MouseEvent) => {
    e.stopPropagation();
    toggleModuleCollapse(module.id);
  };

  return (
    <div
      onClick={onSelect}
      className={`p-3 border-2 rounded-lg cursor-pointer hover:shadow-md transition-all ${
        moduleColors[module.type] || 'bg-gray-50 border-gray-200 text-gray-700'
      }`}
    >
      <div className="flex items-start justify-between gap-2">
        <div className="flex-1 min-w-0">
          <h4 className="text-sm font-semibold truncate">
            {moduleTitles[module.type] || module.type}
          </h4>
          {!module.collapsed && (
            <p className="text-xs mt-1 opacity-75">Click to configure</p>
          )}
        </div>
        <div className="flex items-center gap-1">
          <button
            onClick={handleToggleCollapse}
            className="p-1 hover:bg-white/50 rounded transition"
            title={module.collapsed ? 'Expand' : 'Collapse'}
          >
            {module.collapsed ? (
              <ChevronDown className="w-4 h-4" />
            ) : (
              <ChevronUp className="w-4 h-4" />
            )}
          </button>
          <button
            onClick={handleDelete}
            className="p-1 hover:bg-red-100 rounded transition text-red-600"
            title="Delete"
          >
            <X className="w-4 h-4" />
          </button>
        </div>
      </div>
    </div>
  );
}

