import { X } from 'lucide-react';
import { useInputConfig } from '../../store';
import { ModuleType } from '../../types';
import { SimulationDataGenerator } from './modules/SimulationDataGenerator';
import { SensorTypeSelector } from './modules/SensorTypeSelector';
import { FloorZoneMapper } from './modules/FloorZoneMapper';
import { DataUploadModule } from './modules/DataUploadModule';

export function ModuleDetailsPanel() {
  const { selectedModule, selectModule, updateModule, buildingConfig } = useInputConfig();

  if (!selectedModule) {
    return (
      <div className="w-80 border-l border-gray-200 bg-gray-50 p-6 flex items-center justify-center">
        <div className="text-center text-gray-400">
          <p className="text-sm">Select a module to configure</p>
        </div>
      </div>
    );
  }

  const handleUpdate = (config: any) => {
    updateModule(selectedModule.id, config);
  };

  const availableFloors = buildingConfig.floors.map((f) => ({
    id: f.id,
    name: f.name,
  }));

  const renderModuleConfig = () => {
    switch (selectedModule.type) {
      case ModuleType.Simulation:
        return (
          <SimulationDataGenerator
            config={selectedModule.config}
            onUpdate={handleUpdate}
          />
        );
      case ModuleType.Sensor:
        return (
          <SensorTypeSelector
            config={selectedModule.config}
            onUpdate={handleUpdate}
          />
        );
      case ModuleType.FloorZone:
        return (
          <FloorZoneMapper
            config={selectedModule.config}
            onUpdate={handleUpdate}
            availableFloors={availableFloors}
          />
        );
      case ModuleType.Upload:
        return (
          <DataUploadModule
            config={selectedModule.config}
            onUpdate={handleUpdate}
          />
        );
      default:
        return <p className="text-sm text-gray-500">No configuration available</p>;
    }
  };

  const moduleTitles: Record<string, string> = {
    [ModuleType.Simulation]: 'Simulation Data Generator',
    [ModuleType.Sensor]: 'Sensor Type Selector',
    [ModuleType.FloorZone]: 'Floor & Zone Mapper',
    [ModuleType.Upload]: 'Data Upload Module',
  };

  return (
    <div className="w-80 border-l border-gray-200 bg-white flex flex-col">
      <div className="p-4 border-b border-gray-200 flex items-center justify-between">
        <h3 className="text-sm font-semibold text-gray-800">
          {moduleTitles[selectedModule.type] || 'Module Configuration'}
        </h3>
        <button
          onClick={() => selectModule(null)}
          className="p-1 hover:bg-gray-100 rounded transition"
        >
          <X className="w-4 h-4 text-gray-500" />
        </button>
      </div>
      <div className="flex-1 overflow-y-auto p-6">
        {renderModuleConfig()}
      </div>
    </div>
  );
}

