import { useState, useCallback } from 'react';
import { Upload, File, X, CheckCircle2, AlertCircle, Loader2 } from 'lucide-react';
import { UploadConfig } from '../../types';

interface DataUploadModuleProps {
  config: UploadConfig;
  onUpdate: (config: UploadConfig) => void;
}

export function DataUploadModule({ config, onUpdate }: DataUploadModuleProps) {
  const [files, setFiles] = useState<File[]>(config.files || []);
  const [uploadStatus, setUploadStatus] = useState<'idle' | 'uploading' | 'success' | 'error'>(
    config.uploadStatus || 'idle'
  );
  const [isDragging, setIsDragging] = useState(false);

  const handleDragOver = useCallback((e: React.DragEvent) => {
    e.preventDefault();
    setIsDragging(true);
  }, []);

  const handleDragLeave = useCallback((e: React.DragEvent) => {
    e.preventDefault();
    setIsDragging(false);
  }, []);

  const handleDrop = useCallback(
    (e: React.DragEvent) => {
      e.preventDefault();
      setIsDragging(false);

      const droppedFiles = Array.from(e.dataTransfer.files).filter(
        (file) => file.type === 'text/csv' || file.type === 'application/json' || file.name.endsWith('.csv') || file.name.endsWith('.json')
      );

      if (droppedFiles.length > 0) {
        const newFiles = [...files, ...droppedFiles];
        setFiles(newFiles);
        onUpdate({ ...config, files: newFiles });
      }
    },
    [files, config, onUpdate]
  );

  const handleFileInput = useCallback(
    (e: React.ChangeEvent<HTMLInputElement>) => {
      const selectedFiles = Array.from(e.target.files || []).filter(
        (file) => file.type === 'text/csv' || file.type === 'application/json' || file.name.endsWith('.csv') || file.name.endsWith('.json')
      );

      if (selectedFiles.length > 0) {
        const newFiles = [...files, ...selectedFiles];
        setFiles(newFiles);
        onUpdate({ ...config, files: newFiles });
      }
    },
    [files, config, onUpdate]
  );

  const handleRemoveFile = useCallback(
    (index: number) => {
      const newFiles = files.filter((_, i) => i !== index);
      setFiles(newFiles);
      onUpdate({ ...config, files: newFiles });
    },
    [files, config, onUpdate]
  );

  const handleUpload = useCallback(async () => {
    setUploadStatus('uploading');
    onUpdate({ ...config, uploadStatus: 'uploading' });

    // Simulate upload
    setTimeout(() => {
      setUploadStatus('success');
      onUpdate({ ...config, uploadStatus: 'success' });
    }, 2000);
  }, [config, onUpdate]);

  const getStatusIcon = () => {
    switch (uploadStatus) {
      case 'uploading':
        return <Loader2 className="w-5 h-5 animate-spin text-blue-500" />;
      case 'success':
        return <CheckCircle2 className="w-5 h-5 text-green-500" />;
      case 'error':
        return <AlertCircle className="w-5 h-5 text-red-500" />;
      default:
        return null;
    }
  };

  const getStatusText = () => {
    switch (uploadStatus) {
      case 'uploading':
        return 'Uploading...';
      case 'success':
        return 'Upload successful';
      case 'error':
        return 'Upload failed';
      default:
        return 'Ready to upload';
    }
  };

  return (
    <div className="space-y-6">
      <div
        onDragOver={handleDragOver}
        onDragLeave={handleDragLeave}
        onDrop={handleDrop}
        className={`border-2 border-dashed rounded-lg p-8 text-center transition-all ${
          isDragging
            ? 'border-primary bg-primary/5'
            : 'border-gray-300 hover:border-gray-400 bg-gray-50'
        }`}
      >
        <Upload className={`w-12 h-12 mx-auto mb-4 ${isDragging ? 'text-primary' : 'text-gray-400'}`} />
        <p className="text-sm font-medium text-gray-700 mb-2">
          Drag & drop files here, or click to browse
        </p>
        <p className="text-xs text-gray-500 mb-4">Accepts CSV and JSON files</p>
        <input
          type="file"
          accept=".csv,.json"
          multiple
          onChange={handleFileInput}
          className="hidden"
          id="file-upload"
        />
        <label
          htmlFor="file-upload"
          className="inline-block px-4 py-2 bg-primary text-white rounded-lg hover:bg-primary/90 transition cursor-pointer text-sm font-medium"
        >
          Browse Files
        </label>
      </div>

      {files.length > 0 && (
        <div>
          <label className="block text-sm font-medium text-gray-700 mb-2">Uploaded Files</label>
          <div className="space-y-2">
            {files.map((file, index) => (
              <div
                key={index}
                className="flex items-center gap-3 p-3 bg-white border border-gray-200 rounded-lg"
              >
                <File className="w-5 h-5 text-gray-500" />
                <div className="flex-1 min-w-0">
                  <p className="text-sm font-medium text-gray-700 truncate">{file.name}</p>
                  <p className="text-xs text-gray-500">
                    {(file.size / 1024).toFixed(2)} KB
                  </p>
                </div>
                <button
                  onClick={() => handleRemoveFile(index)}
                  className="p-1 text-red-500 hover:text-red-700 hover:bg-red-50 rounded transition"
                >
                  <X className="w-4 h-4" />
                </button>
              </div>
            ))}
          </div>
        </div>
      )}

      {files.length > 0 && (
        <div>
          <button
            onClick={handleUpload}
            disabled={uploadStatus === 'uploading'}
            className="w-full flex items-center justify-center gap-2 px-4 py-3 bg-gradient-to-r from-primary to-accent text-white rounded-lg hover:shadow-md transition-all duration-300 font-medium disabled:opacity-50 disabled:cursor-not-allowed"
          >
            {getStatusIcon()}
            {uploadStatus === 'uploading' ? 'Uploading...' : 'Upload Files'}
          </button>
          {uploadStatus !== 'idle' && (
            <div className="mt-2 flex items-center gap-2 text-sm text-gray-600">
              {getStatusIcon()}
              <span>{getStatusText()}</span>
            </div>
          )}
        </div>
      )}
    </div>
  );
}

