import { useState } from 'react';
import { Building2, MapPin, Plus, Trash2, GripVertical } from 'lucide-react';
import { FloorZoneConfig, Zone } from '../../types';
import { DndContext, closestCenter, KeyboardSensor, PointerSensor, useSensor, useSensors, DragEndEvent } from '@dnd-kit/core';
import { arrayMove, SortableContext, sortableKeyboardCoordinates, useSortable, verticalListSortingStrategy } from '@dnd-kit/sortable';
import { CSS } from '@dnd-kit/utilities';

interface SortableZoneItemProps {
  zone: Zone;
  onDelete: (id: string) => void;
}

function SortableZoneItem({ zone, onDelete }: SortableZoneItemProps) {
  const { attributes, listeners, setNodeRef, transform, transition, isDragging } = useSortable({
    id: zone.id,
  });

  const style = {
    transform: CSS.Transform.toString(transform),
    transition,
    opacity: isDragging ? 0.5 : 1,
  };

  return (
    <div
      ref={setNodeRef}
      style={style}
      className="flex items-center gap-3 p-3 bg-white border border-gray-200 rounded-lg hover:shadow-md transition"
    >
      <div {...attributes} {...listeners} className="cursor-grab active:cursor-grabbing">
        <GripVertical className="w-5 h-5 text-gray-400" />
      </div>
      <MapPin className="w-5 h-5 text-gray-600" />
      <span className="flex-1 text-sm font-medium text-gray-700">{zone.name}</span>
      <button
        onClick={() => onDelete(zone.id)}
        className="p-1 text-red-500 hover:text-red-700 hover:bg-red-50 rounded transition"
      >
        <Trash2 className="w-4 h-4" />
      </button>
    </div>
  );
}

interface FloorZoneMapperProps {
  config: FloorZoneConfig;
  onUpdate: (config: FloorZoneConfig) => void;
  availableFloors: { id: string; name: string }[];
}

export function FloorZoneMapper({ config, onUpdate, availableFloors }: FloorZoneMapperProps) {
  const [selectedFloor, setSelectedFloor] = useState(config.selectedFloor || availableFloors[0]?.id || '');
  const [zones, setZones] = useState<Zone[]>(config.zones || []);
  const [newZoneName, setNewZoneName] = useState('');

  const dndSensors = useSensors(
    useSensor(PointerSensor),
    useSensor(KeyboardSensor, {
      coordinateGetter: sortableKeyboardCoordinates,
    })
  );

  const handleFloorChange = (floorId: string) => {
    setSelectedFloor(floorId);
    const filteredZones = zones.filter((z) => z.floorId === floorId);
    setZones(filteredZones);
    onUpdate({ ...config, selectedFloor: floorId, zones: filteredZones });
  };

  const handleAddZone = () => {
    if (!newZoneName.trim()) return;
    const newZone: Zone = {
      id: `zone-${Date.now()}`,
      name: newZoneName.trim(),
      floorId: selectedFloor,
    };
    const updated = [...zones, newZone];
    setZones(updated);
    setNewZoneName('');
    onUpdate({ ...config, zones: updated });
  };

  const handleDeleteZone = (id: string) => {
    const updated = zones.filter((z) => z.id !== id);
    setZones(updated);
    onUpdate({ ...config, zones: updated });
  };

  const handleDragEnd = (event: DragEndEvent) => {
    const { active, over } = event;
    if (over && active.id !== over.id) {
      const oldIndex = zones.findIndex((z) => z.id === active.id);
      const newIndex = zones.findIndex((z) => z.id === over.id);
      const updated = arrayMove(zones, oldIndex, newIndex);
      setZones(updated);
      onUpdate({ ...config, zones: updated });
    }
  };

  const floorZones = zones.filter((z) => z.floorId === selectedFloor);

  return (
    <div className="space-y-6">
      <div>
        <label className="block text-sm font-medium text-gray-700 mb-2">
          <Building2 className="w-4 h-4 inline mr-2" />
          Select Floor
        </label>
        <select
          value={selectedFloor}
          onChange={(e) => handleFloorChange(e.target.value)}
          className="w-full px-4 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-primary focus:border-transparent"
        >
          {availableFloors.map((floor) => (
            <option key={floor.id} value={floor.id}>
              {floor.name}
            </option>
          ))}
        </select>
      </div>

      <div>
        <div className="flex items-center gap-2 mb-3">
          <MapPin className="w-4 h-4 text-gray-500" />
          <label className="text-sm font-medium text-gray-700">Zones</label>
        </div>
        <div className="flex gap-2 mb-3">
          <input
            type="text"
            value={newZoneName}
            onChange={(e) => setNewZoneName(e.target.value)}
            onKeyPress={(e) => e.key === 'Enter' && handleAddZone()}
            placeholder="Enter zone name..."
            className="flex-1 px-4 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-primary focus:border-transparent"
          />
          <button
            onClick={handleAddZone}
            className="px-4 py-2 bg-primary text-white rounded-lg hover:bg-primary/90 transition flex items-center gap-2"
          >
            <Plus className="w-4 h-4" />
            Add
          </button>
        </div>

        {floorZones.length > 0 ? (
          <DndContext
            sensors={dndSensors}
            collisionDetection={closestCenter}
            onDragEnd={handleDragEnd}
          >
            <SortableContext items={floorZones.map((z) => z.id)} strategy={verticalListSortingStrategy}>
              <div className="space-y-2">
                {floorZones.map((zone) => (
                  <SortableZoneItem key={zone.id} zone={zone} onDelete={handleDeleteZone} />
                ))}
              </div>
            </SortableContext>
          </DndContext>
        ) : (
          <div className="p-8 text-center text-gray-400 border-2 border-dashed border-gray-300 rounded-lg">
            <MapPin className="w-8 h-8 mx-auto mb-2 opacity-50" />
            <p className="text-sm">No zones added yet</p>
          </div>
        )}
      </div>
    </div>
  );
}

