import React, { ReactNode } from 'react';
import { DndContext, DragEndEvent, DragOverlay, DragStartEvent, PointerSensor, useSensor, useSensors } from '@dnd-kit/core';
import { useInputConfig } from '../store';
import { ModuleType, GridPosition } from '../types';

interface DndProviderProps {
  children: ReactNode;
  dragOverlay?: ReactNode;
}

export function DndProvider({ children, dragOverlay }: DndProviderProps) {
  const { addModule } = useInputConfig();
  const [activeId, setActiveId] = React.useState<string | null>(null);
  const [draggedType, setDraggedType] = React.useState<ModuleType | null>(null);

  const sensors = useSensors(
    useSensor(PointerSensor, {
      activationConstraint: {
        distance: 8,
      },
    })
  );

  const handleDragStart = (event: DragStartEvent) => {
    setActiveId(event.active.id as string);
    const data = event.active.data.current;
    if (data?.type) {
      setDraggedType(data.type as ModuleType);
    }
  };

  const handleDragEnd = (event: DragEndEvent) => {
    const { active, over } = event;
    setActiveId(null);
    setDraggedType(null);

    if (!over) return;

    const overData = over.data.current;
    if (overData?.floorId && overData?.position) {
      const activeData = active.data.current;
      if (activeData?.type) {
        addModule(
          overData.floorId,
          overData.position as GridPosition,
          activeData.type as ModuleType
        );
      }
    }
  };

  return (
    <DndContext
      sensors={sensors}
      onDragStart={handleDragStart}
      onDragEnd={handleDragEnd}
    >
      {children}
      <DragOverlay>
        {activeId && draggedType && dragOverlay}
      </DragOverlay>
    </DndContext>
  );
}

